// SAFE Drainer Simulation Server (Express + Socket.io)
// WARNING: This project is a SAFE TRAINING TOOL. No code here performs real draining,
// no approvals/transfers/permits are executed. Every "DANGER ZONE" is intentionally disabled.
const express = require('express');
const bodyParser = require('body-parser');
const fs = require('fs');
const http = require('http');
const { Server } = require('socket.io');
const path = require('path');

const approvalsHandler = require('./handlers/approvals');
const permitsHandler = require('./handlers/permits');
const signaturesHandler = require('./handlers/signatures');
const simulateDrain = require('./handlers/simulateDrain');

const app = express();
const server = http.createServer(app);
const io = new Server(server);

app.use(bodyParser.json());
app.use(express.static(path.join(__dirname, '..', 'frontend')));

function logEvent(event) {
  const line = JSON.stringify(event) + "\n";
  fs.appendFileSync(path.join(__dirname, 'logs', 'events.log'), line);
  // emit event to dashboard
  io.emit('event', event);
}

// Ensure logs dir exists
if (!fs.existsSync(path.join(__dirname, 'logs'))) {
  fs.mkdirSync(path.join(__dirname, 'logs'), { recursive: true });
}

app.post('/connectWallet', (req, res) => {
  const { address, sessionId } = req.body;
  const event = { type: 'wallet_connect', address, sessionId: sessionId || null, timestamp: Date.now() };
  logEvent(event);
  res.json({ status: 'connected' });
});

app.post('/approve', (req, res) => {
  const event = approvalsHandler(req.body);
  logEvent(event);
  res.json({ status: 'approval_requested' });
});

app.post('/permit', (req, res) => {
  const event = permitsHandler(req.body);
  logEvent(event);
  res.json({ status: 'permit_requested' });
});

app.post('/sign', (req, res) => {
  const event = signaturesHandler(req.body);
  logEvent(event);
  res.json({ status: 'signature_requested' });
});

app.post('/simulateDrain', (req, res) => {
  const event = simulateDrain(req.body);
  logEvent(event);
  res.json({ status: 'drain_simulated' });
});

app.get('/logs', (req, res) => {
  const logs = fs.readFileSync(path.join(__dirname, 'logs', 'events.log'), 'utf-8');
  res.type('text').send(logs);
});

app.get('/logs/json', (req, res) => {
  try {
    const raw = fs.readFileSync(path.join(__dirname, 'logs', 'events.log'), 'utf-8').trim().split('\n').map(l => JSON.parse(l));
    res.json(raw);
  } catch (err) {
    res.json([]);
  }
});

// Simple sessions endpoint for per-wallet session tracking (safe, client-controlled)
app.get('/sessions', (req, res) => {
  // This is a placeholder - in real deployments you'd track sessions in a DB
  res.json({ note: 'Session tracking is simulated. See README for how to extend.' });
});

// Socket.io connection for live updates and charts
io.on('connection', socket => {
  console.log('dashboard connected:', socket.id);
  socket.on('requestRecent', () => {
    try {
      const raw = fs.readFileSync(path.join(__dirname, 'logs', 'events.log'), 'utf-8').trim();
      const lines = raw ? raw.split('\n').map(l => JSON.parse(l)) : [];
      socket.emit('recent', lines.slice(-200));
    } catch (e) {
      socket.emit('recent', []);
    }
  });
});

const PORT = process.env.PORT || 3000;
server.listen(PORT, () => console.log('Safe Drainer Simulation running on http://localhost:' + PORT));
